<?php
// app/Http/Controllers/LaporanController.php

namespace App\Http\Controllers;

use App\Models\Laporan;
use App\Models\Desa;
use App\Models\Dusun;
use App\Models\Notifikasi;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class LaporanController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }
    
    /**
     * Display laporan list
     */
    public function index()
    {
        $user = auth()->user();
        
        $laporanQuery = Laporan::with('user', 'desa', 'dusun')
            ->orderBy('created_at', 'desc');
        
        // Filter by role
        if ($user->role === 'warga') {
            $laporanQuery->where('user_id', $user->id);
        } elseif ($user->role === 'pemdes' && $user->desa_id) {
            $laporanQuery->where('desa_id', $user->desa_id);
        }
        
        $laporan = $laporanQuery->paginate(15);
        
        // Get desa & dusun for form
        $desa = Desa::all();
        $dusun = $user->desa_id ? Dusun::where('desa_id', $user->desa_id)->get() : collect();
        
        return view('dashboard.laporan', compact('laporan', 'desa', 'dusun'));
    }
    
    /**
     * Store new laporan
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'desa_id' => 'required|exists:desa,id',
            'dusun_id' => 'nullable|exists:dusun,id',
            'jenis_laporan' => 'required|in:sumur_kering,mata_air_kering,kebutuhan_air,lainnya',
            'deskripsi' => 'required|string|min:10',
            'foto' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',
        ]);
        
        $validated['user_id'] = auth()->id();
        
        // Upload foto
        if ($request->hasFile('foto')) {
            $file = $request->file('foto');
            $filename = time() . '_' . $file->getClientOriginalName();
            $path = $file->storeAs('laporan', $filename, 'public');
            $validated['foto'] = $path;
        }
        
        $laporan = Laporan::create($validated);
        
        // Create notification for admin/pemdes
        $this->createLaporanNotification($laporan);
        
        return redirect()->back()->with('success', 'Laporan berhasil dikirim! Akan segera ditinjau.');
    }
    
    /**
     * Update laporan status (Admin/Pemdes only)
     */
    public function updateStatus(Request $request, $id)
    {
        $validated = $request->validate([
            'status' => 'required|in:pending,ditinjau,selesai',
        ]);
        
        $laporan = Laporan::findOrFail($id);
        
        // Check authorization
        if (!auth()->user()->isAdmin() && !auth()->user()->isPemdes()) {
            return redirect()->back()->with('error', 'Anda tidak memiliki akses.');
        }
        
        $laporan->update($validated);
        
        // Notify user
        Notifikasi::create([
            'user_id' => $laporan->user_id,
            'judul' => 'Status Laporan Diperbarui',
            'pesan' => "Laporan Anda tentang {$laporan->jenis_laporan} telah diubah menjadi: {$validated['status']}",
            'tipe' => 'info',
        ]);
        
        return redirect()->back()->with('success', 'Status laporan berhasil diperbarui!');
    }
    
    /**
     * Delete laporan
     */
    public function destroy($id)
    {
        $laporan = Laporan::findOrFail($id);
        
        // Check authorization
        if ($laporan->user_id !== auth()->id() && !auth()->user()->isAdmin()) {
            return redirect()->back()->with('error', 'Anda tidak bisa menghapus laporan ini.');
        }
        
        // Delete foto if exists
        if ($laporan->foto) {
            Storage::disk('public')->delete($laporan->foto);
        }
        
        $laporan->delete();
        
        return redirect()->back()->with('success', 'Laporan berhasil dihapus!');
    }
    
    /**
     * Show laporan detail
     */
    public function show($id)
    {
        $laporan = Laporan::with('user', 'desa', 'dusun')->findOrFail($id);
        
        return view('dashboard.laporan-detail', compact('laporan'));
    }
    
    /**
     * Get dusun by desa (AJAX)
     */
    public function getDusun($desaId)
    {
        $dusun = Dusun::where('desa_id', $desaId)->get();
        return response()->json($dusun);
    }
    
    /**
     * Create notification for new laporan
     */
    protected function createLaporanNotification($laporan)
    {
        $tipe = match($laporan->jenis_laporan) {
            'sumur_kering', 'mata_air_kering' => 'peringatan',
            'kebutuhan_air' => 'penting',
            default => 'info',
        };
        
        // Notify pemdes of the desa
        Notifikasi::create([
            'desa_id' => $laporan->desa_id,
            'judul' => 'Laporan Baru dari Warga',
            'pesan' => "Laporan baru: {$laporan->jenis_laporan} dari {$laporan->user->name}",
            'tipe' => $tipe,
        ]);
    }
}
